<?php
require_once '../includes/config.php';
require_once '../includes/db.php';
require_once '../includes/functions.php';

// Redirect if not logged in
if (!isLoggedIn()) {
    setFlashMessage('danger', 'Please log in to access this page');
    redirect(APP_URL . '/pages/login.php');
}

$pageTitle = 'Manage Book Links';
$userId = $_SESSION['user_id'];
$errors = [];
$success = '';

// Check if book ID is provided
if (!isset($_GET['book_id'])) {
    setFlashMessage('danger', 'Book ID is required');
    redirect(APP_URL . '/pages/dashboard.php');
}

$bookId = (int)$_GET['book_id'];
$book = getBookById($bookId);

// Verify book exists and belongs to current user
if (!$book || $book['user_id'] != $userId) {
    setFlashMessage('danger', 'Book not found or you do not have permission to edit it');
    redirect(APP_URL . '/pages/dashboard.php');
}

// Get existing buttons
$buttons = getButtonsByBookId($bookId);

// Handle button type options
$buttonTypes = ['ebook', 'paperback', 'hardcover', 'audiobook', 'other'];

// Handle form submission for adding/editing a button
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    if ($_POST['action'] === 'add' || $_POST['action'] === 'edit') {
        $buttonId = isset($_POST['button_id']) ? (int)$_POST['button_id'] : null;
        $label = sanitize($_POST['label']);
        $url = sanitize($_POST['url']);
        $buttonType = sanitize($_POST['button_type']);
        $position = isset($_POST['position']) ? (int)$_POST['position'] : count($buttons);
        
        // Validate input
        if (empty($label)) {
            $errors[] = 'Button label is required';
        }
        
        if (empty($url)) {
            $errors[] = 'Button URL is required';
        } elseif (!filter_var($url, FILTER_VALIDATE_URL)) {
            $errors[] = 'Invalid URL format';
        }
        
        if (!in_array($buttonType, $buttonTypes)) {
            $buttonType = 'other';
        }
        
        // If no errors, save button
        if (empty($errors)) {
            $buttonData = [
                'label' => $label,
                'url' => $url,
                'button_type' => $buttonType,
                'position' => $position
            ];
            
            if ($buttonId) {
                // Update existing button
                $button = $db->selectOne("SELECT * FROM buttons WHERE id = ? AND book_id = ?", [$buttonId, $bookId]);
                
                if ($button) {
                    $db->update('buttons', $buttonData, 'id = ?', [$buttonId]);
                    $success = 'Button updated successfully';
                }
            } else {
                // Add new button
                $buttonData['book_id'] = $bookId;
                $db->insert('buttons', $buttonData);
                $success = 'Button added successfully';
            }
            
            // Refresh the buttons list
            $buttons = getButtonsByBookId($bookId);
        }
    } elseif ($_POST['action'] === 'delete' && isset($_POST['button_id'])) {
        $buttonId = (int)$_POST['button_id'];
        
        // Verify button belongs to current book
        $button = $db->selectOne("SELECT * FROM buttons WHERE id = ? AND book_id = ?", [$buttonId, $bookId]);
        
        if ($button) {
            $db->delete('buttons', 'id = ?', [$buttonId]);
            $success = 'Button deleted successfully';
            
            // Refresh the buttons list
            $buttons = getButtonsByBookId($bookId);
        }
    } elseif ($_POST['action'] === 'reorder' && isset($_POST['button_order'])) {
        $buttonOrder = explode(',', $_POST['button_order']);
        
        foreach ($buttonOrder as $position => $buttonId) {
            $db->update('buttons', ['position' => $position], 'id = ? AND book_id = ?', [(int)$buttonId, $bookId]);
        }
        
        $success = 'Button order updated successfully';
        
        // Refresh the buttons list
        $buttons = getButtonsByBookId($bookId);
    }
}

include '../templates/header.php';
?>

<div class="row mb-4">
    <div class="col">
        <h1>Manage Book Links: <?php echo sanitize($book['title']); ?></h1>
        <p>Add and organize purchase links for your book.</p>
    </div>
    <div class="col-auto">
        <a href="<?php echo APP_URL; ?>/pages/edit-book.php?id=<?php echo $bookId; ?>" class="btn btn-secondary">Back to Book</a>
        <a href="<?php echo APP_URL . '/' . getUserById($userId)['permalink']; ?>" target="_blank" class="btn btn-success">Preview Page</a>
    </div>
</div>

<div class="row">
    <div class="col-md-8">
        <?php if (!empty($errors)): ?>
            <div class="alert alert-danger">
                <ul class="mb-0">
                    <?php foreach ($errors as $error): ?>
                        <li><?php echo $error; ?></li>
                    <?php endforeach; ?>
                </ul>
            </div>
        <?php endif; ?>
        
        <?php if (!empty($success)): ?>
            <div class="alert alert-success">
                <?php echo $success; ?>
            </div>
        <?php endif; ?>
        
        <div class="card mb-4">
            <div class="card-header">
                <h3>Add New Link</h3>
            </div>
            <div class="card-body">
                <form method="POST">
                    <input type="hidden" name="action" value="add">
                    <div class="row mb-3">
                        <div class="col-md-6">
                            <label for="label" class="form-label">Button Label *</label>
                            <input type="text" id="label" name="label" class="form-control" required>
                            <small class="form-text text-muted">e.g., "Buy on Amazon", "Apple Books", etc.</small>
                        </div>
                        <div class="col-md-6">
                            <label for="url" class="form-label">Button URL *</label>
                            <input type="url" id="url" name="url" class="form-control" required>
                            <small class="form-text text-muted">The purchase link</small>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="button_type" class="form-label">Button Type</label>
                        <select id="button_type" name="button_type" class="form-select">
                            <?php foreach ($buttonTypes as $type): ?>
                                <option value="<?php echo $type; ?>"><?php echo ucfirst($type); ?></option>
                            <?php endforeach; ?>
                        </select>
                        <small class="form-text text-muted">Categorize your link</small>
                    </div>
                    
                    <div class="d-grid">
                        <button type="submit" class="btn btn-primary">Add Link</button>
                    </div>
                </form>
            </div>
        </div>
        
        <div class="card">
            <div class="card-header">
                <h3>Book Links</h3>
            </div>
            <div class="card-body">
                <?php if (empty($buttons)): ?>
                    <div class="alert alert-info">
                        You haven't added any links yet. Use the form above to add your first link.
                    </div>
                <?php else: ?>
                    <p>Drag and drop to reorder your links. Links will be grouped by type on your book page.</p>
                    <form method="POST" id="reorderForm">
                        <input type="hidden" name="action" value="reorder">
                        <input type="hidden" name="button_order" id="buttonOrder">
                    </form>
                    <div class="list-group" id="sortableButtons">
                        <?php foreach ($buttons as $button): ?>
                            <div class="list-group-item list-group-item-action d-flex justify-content-between align-items-center" data-id="<?php echo $button['id']; ?>">
                                <div>
                                    <h5 class="mb-1"><?php echo sanitize($button['label']); ?></h5>
                                    <p class="mb-1 text-muted small"><?php echo sanitize($button['url']); ?></p>
                                    <span class="badge bg-info"><?php echo ucfirst(sanitize($button['button_type'])); ?></span>
                                </div>
                                <div class="btn-group">
                                    <button type="button" class="btn btn-sm btn-outline-primary edit-button" 
                                            data-id="<?php echo $button['id']; ?>"
                                            data-label="<?php echo sanitize($button['label']); ?>"
                                            data-url="<?php echo sanitize($button['url']); ?>"
                                            data-type="<?php echo sanitize($button['button_type']); ?>"
                                            data-bs-toggle="modal" data-bs-target="#editButtonModal">
                                        Edit
                                    </button>
                                    <form method="POST" class="d-inline">
                                        <input type="hidden" name="action" value="delete">
                                        <input type="hidden" name="button_id" value="<?php echo $button['id']; ?>">
                                        <button type="submit" class="btn btn-sm btn-outline-danger" onclick="return confirm('Are you sure you want to delete this link?')">Delete</button>
                                    </form>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                    <div class="mt-3">
                        <button type="button" id="saveOrder" class="btn btn-success">Save Order</button>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <div class="col-md-4">
        <div class="card">
            <div class="card-header">
                <h3>Tips</h3>
            </div>
            <div class="card-body">
                <ul>
                    <li>Add links to all platforms where your book is available.</li>
                    <li>Use clear, recognizable names for each button.</li>
                    <li>Organize buttons by type (ebook, paperback, etc.) for better user experience.</li>
                    <li>Consider adding special links for signed copies or limited editions.</li>
                    <li>Check all links to make sure they work properly.</li>
                </ul>
            </div>
        </div>
    </div>
</div>

<!-- Edit Button Modal -->
<div class="modal fade" id="editButtonModal" tabindex="-1" aria-labelledby="editButtonModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="editButtonModalLabel">Edit Link</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form method="POST">
                <div class="modal-body">
                    <input type="hidden" name="action" value="edit">
                    <input type="hidden" name="button_id" id="editButtonId">
                    
                    <div class="mb-3">
                        <label for="editLabel" class="form-label">Button Label *</label>
                        <input type="text" id="editLabel" name="label" class="form-control" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="editUrl" class="form-label">Button URL *</label>
                        <input type="url" id="editUrl" name="url" class="form-control" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="editButtonType" class="form-label">Button Type</label>
                        <select id="editButtonType" name="button_type" class="form-select">
                            <?php foreach ($buttonTypes as $type): ?>
                                <option value="<?php echo $type; ?>"><?php echo ucfirst($type); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Save Changes</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Add JavaScript for drag and drop functionality -->
<script src="https://cdn.jsdelivr.net/npm/sortablejs@1.14.0/Sortable.min.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Initialize sortable list
    var sortableList = document.getElementById('sortableButtons');
    if (sortableList) {
        new Sortable(sortableList, {
            animation: 150,
            ghostClass: 'bg-light'
        });
    }
    
    // Save button order
    var saveOrderBtn = document.getElementById('saveOrder');
    if (saveOrderBtn) {
        saveOrderBtn.addEventListener('click', function() {
            var buttons = document.querySelectorAll('#sortableButtons .list-group-item');
            var order = [];
            
            buttons.forEach(function(button) {
                order.push(button.getAttribute('data-id'));
            });
            
            document.getElementById('buttonOrder').value = order.join(',');
            document.getElementById('reorderForm').submit();
        });
    }
    
    // Populate edit modal
    var editButtons = document.querySelectorAll('.edit-button');
    editButtons.forEach(function(button) {
        button.addEventListener('click', function() {
            var id = this.getAttribute('data-id');
            var label = this.getAttribute('data-label');
            var url = this.getAttribute('data-url');
            var type = this.getAttribute('data-type');
            
            document.getElementById('editButtonId').value = id;
            document.getElementById('editLabel').value = label;
            document.getElementById('editUrl').value = url;
            document.getElementById('editButtonType').value = type;
        });
    });
});
</script>

<?php include '../templates/footer.php'; ?>